
import os
import argparse
from datetime import datetime, date
from uuid import uuid4

from flask import (
    Flask, render_template, redirect, url_for, request, flash,
    send_from_directory, abort
)
from flask_sqlalchemy import SQLAlchemy
from flask_login import (
    LoginManager, login_user, login_required, logout_user,
    current_user, UserMixin
)
from werkzeug.security import generate_password_hash, check_password_hash
from werkzeug.utils import secure_filename

# ---------------- Config ----------------
BASE_DIR = os.path.abspath(os.path.dirname(__file__))
UPLOAD_FOLDER = os.path.join(BASE_DIR, "uploads")
ALLOWED_EXT = {"png", "jpg", "jpeg", "gif", "webp"}

app = Flask(__name__)
app.config["SECRET_KEY"] = os.environ.get("SECRET_KEY", "dev-secret-change-me")
app.config["SQLALCHEMY_DATABASE_URI"] = "sqlite:///" + os.path.join(BASE_DIR, "app.db")
app.config["SQLALCHEMY_TRACK_MODIFICATIONS"] = False
app.config["UPLOAD_FOLDER"] = UPLOAD_FOLDER

db = SQLAlchemy(app)
login_manager = LoginManager(app)
login_manager.login_view = "login"

# ---------------- Models ----------------
class User(db.Model, UserMixin):
    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(120), nullable=False)
    email = db.Column(db.String(120), unique=True, nullable=False, index=True)
    gender = db.Column(db.String(20))               # jenis kelamin
    phone = db.Column(db.String(50))                # no hp
    position = db.Column(db.String(120))            # jabatan
    photo = db.Column(db.String(255))               # filename
    role = db.Column(db.String(20), default="user") # user/admin
    password_hash = db.Column(db.String(255), nullable=False)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)

    loans = db.relationship("Loan", backref="user", lazy=True)

    def set_password(self, password):
        self.password_hash = generate_password_hash(password)

    def check_password(self, password):
        return check_password_hash(self.password_hash, password)

class Item(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(200), nullable=False)         # nama barang
    completeness = db.Column(db.Text)                        # kelengkapan
    quantity = db.Column(db.Integer, default=1)              # jumlah
    active = db.Column(db.Boolean, default=True)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)

    loans = db.relationship("Loan", backref="item", lazy=True)

class Loan(db.Model):
    __tablename__ = "loan"
    id = db.Column(db.Integer, primary_key=True)
    user_id = db.Column(db.Integer, db.ForeignKey("user.id"), nullable=False)
    item_id = db.Column(db.Integer, db.ForeignKey("item.id"), nullable=False)

    start_date = db.Column(db.Date)      # tgl pinjam (rencana/aktual)
    end_date = db.Column(db.Date)        # tgl kembali (rencana/aktual)

    request_photo = db.Column(db.String(255))  # foto saat pengajuan (opsional)
    borrow_photo = db.Column(db.String(255))   # foto pas meminjam
    return_photo = db.Column(db.String(255))   # foto pas pengembalian

    status = db.Column(db.String(20), default="requested")  # requested, approved, borrowed, completed
    created_at = db.Column(db.DateTime, default=datetime.utcnow)

    def status_label(self):
        return {
            "requested": "Diajukan",
            "approved": "Disetujui",
            "borrowed": "Dipinjam",
            "completed": "Selesai",
        }.get(self.status, self.status)

# ---------------- Helpers ----------------
def allowed_file(filename):
    return "." in filename and filename.rsplit(".", 1)[1].lower() in ALLOWED_EXT

def save_upload(file, prefix=""):
    if not file or file.filename == "":
        return None
    if not allowed_file(file.filename):
        return None
    os.makedirs(app.config["UPLOAD_FOLDER"], exist_ok=True)
    ext = file.filename.rsplit(".", 1)[1].lower()
    fname = f"{prefix}{uuid4().hex}.{ext}"
    path = os.path.join(app.config["UPLOAD_FOLDER"], secure_filename(fname))
    file.save(path)
    return fname

def admin_required(func):
    from functools import wraps
    @wraps(func)
    def wrapper(*args, **kwargs):
        if not current_user.is_authenticated or current_user.role != "admin":
            abort(403)
        return func(*args, **kwargs)
    return wrapper

def count_active_loans(item_id):
    """Count active reservations for an item (approved + borrowed)."""
    return Loan.query.filter(
        Loan.item_id == item_id,
        Loan.status.in_(["approved", "borrowed"])
    ).count()

def available_count(item):
    used = count_active_loans(item.id)
    return max(0, (item.quantity or 0) - used)

app.jinja_env.globals.update(available_count=available_count)

@login_manager.user_loader
def load_user(user_id):
    return User.query.get(int(user_id))

# ---------------- Routes: auth ----------------
@app.route("/login", methods=["GET", "POST"])
def login():
    if request.method == "POST":
        email = request.form.get("email", "").strip().lower()
        password = request.form.get("password", "")
        user = User.query.filter_by(email=email).first()
        if user and user.check_password(password):
            login_user(user)
            return redirect(url_for("dashboard"))
        flash("Email atau password salah.", "danger")
    return render_template("login.html")

@app.route("/register", methods=["GET", "POST"])
def register():
    if request.method == "POST":
        name = request.form.get("name", "").strip()
        email = request.form.get("email", "").strip().lower()
        gender = request.form.get("gender", "")
        phone = request.form.get("phone", "")
        position = request.form.get("position", "")
        password = request.form.get("password", "")
        photo_file = request.files.get("photo")

        if not name or not email or not password:
            flash("Nama, email, dan password wajib diisi.", "danger")
            return render_template("register.html")

        if User.query.filter_by(email=email).first():
            flash("Email sudah terdaftar.", "danger")
            return render_template("register.html")

        user = User(
            name=name, email=email, gender=gender, phone=phone,
            position=position, role="user"
        )
        user.set_password(password)
        if photo_file and allowed_file(photo_file.filename):
            user.photo = save_upload(photo_file, prefix="user-")
        db.session.add(user)
        db.session.commit()
        flash("Registrasi berhasil. Silakan login.", "success")
        return redirect(url_for("login"))
    return render_template("register.html")

@app.route("/logout")
@login_required
def logout():
    logout_user()
    return redirect(url_for("login"))

# ---------------- Routes: common ----------------
@app.route("/")
def index():
    if current_user.is_authenticated:
        return redirect(url_for("dashboard"))
    return redirect(url_for("login"))

@app.route("/dashboard")
@login_required
def dashboard():
    items = Item.query.filter_by(active=True).all()
    loans = Loan.query.order_by(Loan.created_at.desc()).limit(5).all() if current_user.role == "admin" else Loan.query.filter_by(user_id=current_user.id).order_by(Loan.created_at.desc()).limit(5).all()
    any_borrowed = Loan.query.filter_by(user_id=current_user.id, status="borrowed").count() > 0
    return render_template("dashboard.html", items=items, loans=loans, any_borrowed=any_borrowed, available_count=available_count)

@app.route("/uploads/<path:filename>")
def uploads(filename):
    return send_from_directory(app.config["UPLOAD_FOLDER"], filename)

# ---------------- Routes: User Loan Flow ----------------
@app.route("/items/ready")
@login_required
def items_ready():
    items = Item.query.filter_by(active=True).all()
    ready = [(it, available_count(it)) for it in items if available_count(it) > 0]
    return render_template("items_ready.html", ready=ready)

@app.route("/loans/request", methods=["GET", "POST"])
@login_required
def loan_request():
    items = Item.query.filter_by(active=True).all()
    ready_items = [it for it in items if available_count(it) > 0]

    if request.method == "POST":
        item_id = request.form.get("item_id")
        start_date = request.form.get("start_date")
        end_date = request.form.get("end_date")
        request_photo_file = request.files.get("request_photo")

        if not item_id:
            flash("Pilih barang terlebih dahulu.", "danger")
            return render_template("loan_request.html", items=ready_items)

        item = Item.query.get(int(item_id))
        if not item or available_count(item) <= 0:
            flash("Barang tidak tersedia.", "danger")
            return render_template("loan_request.html", items=ready_items)

        loan = Loan(
            user_id=current_user.id,
            item_id=item.id,
            start_date=datetime.strptime(start_date, "%Y-%m-%d").date() if start_date else None,
            end_date=datetime.strptime(end_date, "%Y-%m-%d").date() if end_date else None,
            status="requested",
        )
        if request_photo_file and allowed_file(request_photo_file.filename):
            loan.request_photo = save_upload(request_photo_file, prefix="req-")
        db.session.add(loan)
        db.session.commit()
        flash("Pengajuan peminjaman berhasil dikirim.", "success")
        return redirect(url_for("my_loans"))
    return render_template("loan_request.html", items=ready_items)

@app.route("/my-loans")
@login_required
def my_loans():
    loans = Loan.query.filter_by(user_id=current_user.id).order_by(Loan.created_at.desc()).all()
    return render_template("my_loans.html", loans=loans)

@app.route("/loans/<int:loan_id>/return", methods=["POST"])
@login_required
def loan_return(loan_id):
    loan = Loan.query.get_or_404(loan_id)
    if loan.user_id != current_user.id and current_user.role != "admin":
        abort(403)
    if loan.status != "borrowed":
        flash("Pinjaman tidak dalam status 'Dipinjam'.", "danger")
        return redirect(url_for("my_loans"))
    ret_photo = request.files.get("return_photo")
    if not ret_photo or not allowed_file(ret_photo.filename):
        flash("Foto pengembalian wajib dan harus gambar yang valid.", "danger")
        return redirect(url_for("my_loans"))
    loan.return_photo = save_upload(ret_photo, prefix="ret-")
    loan.status = "completed"
    loan.end_date = date.today()
    db.session.commit()
    flash("Pengembalian berhasil, terima kasih.", "success")
    return redirect(url_for("my_loans"))

# ---------------- Routes: Admin - Items ----------------
@app.route("/admin")
@login_required
@admin_required
def admin_dashboard():
    total_users = User.query.count()
    total_items = Item.query.count()
    total_loans = Loan.query.count()
    return render_template("admin_dashboard.html",
        total_users=total_users, total_items=total_items, total_loans=total_loans
    )

@app.route("/admin/items")
@login_required
@admin_required
def admin_items():
    items = Item.query.order_by(Item.created_at.desc()).all()
    return render_template("items_index.html", items=items, available_count=available_count)

@app.route("/admin/items/create", methods=["GET", "POST"])
@login_required
@admin_required
def admin_items_create():
    if request.method == "POST":
        name = request.form.get("name", "").strip()
        completeness = request.form.get("completeness", "")
        quantity = int(request.form.get("quantity") or 0)
        active = True if request.form.get("active") == "on" else False
        if not name:
            flash("Nama barang wajib diisi.", "danger")
            return render_template("item_form.html", item=None)
        it = Item(name=name, completeness=completeness, quantity=quantity, active=active)
        db.session.add(it)
        db.session.commit()
        flash("Barang berhasil ditambahkan.", "success")
        return redirect(url_for("admin_items"))
    return render_template("item_form.html", item=None)

@app.route("/admin/items/<int:item_id>/edit", methods=["GET", "POST"])
@login_required
@admin_required
def admin_items_edit(item_id):
    it = Item.query.get_or_404(item_id)
    if request.method == "POST":
        it.name = request.form.get("name", "").strip()
        it.completeness = request.form.get("completeness", "")
        it.quantity = int(request.form.get("quantity") or 0)
        it.active = True if request.form.get("active") == "on" else False
        db.session.commit()
        flash("Barang berhasil diperbarui.", "success")
        return redirect(url_for("admin_items"))
    return render_template("item_form.html", item=it)

@app.route("/admin/items/<int:item_id>/delete", methods=["POST"])
@login_required
@admin_required
def admin_items_delete(item_id):
    it = Item.query.get_or_404(item_id)
    has_loans = Loan.query.filter_by(item_id=it.id).first() is not None
    if has_loans:
        flash("Tidak dapat menghapus barang yang sudah memiliki riwayat pinjaman.", "danger")
        return redirect(url_for("admin_items"))
    db.session.delete(it)
    db.session.commit()
    flash("Barang berhasil dihapus.", "success")
    return redirect(url_for("admin_items"))

# ---------------- Routes: Admin - Users ----------------
@app.route("/admin/users")
@login_required
@admin_required
def admin_users():
    users = User.query.order_by(User.created_at.desc()).all()
    return render_template("users_index.html", users=users)

@app.route("/admin/users/create", methods=["GET", "POST"])
@login_required
@admin_required
def admin_users_create():
    if request.method == "POST":
        name = request.form.get("name", "").strip()
        email = request.form.get("email", "").strip().lower()
        gender = request.form.get("gender", "")
        phone = request.form.get("phone", "")
        position = request.form.get("position", "")
        role = request.form.get("role", "user")
        password = request.form.get("password", "")
        photo_file = request.files.get("photo")

        if not name or not email or not password:
            flash("Nama, email, dan password wajib diisi.", "danger")
            return render_template("user_form.html", user=None)

        if User.query.filter_by(email=email).first():
            flash("Email sudah terdaftar.", "danger")
            return render_template("user_form.html", user=None)

        user = User(name=name, email=email, gender=gender, phone=phone, position=position, role=role)
        user.set_password(password)
        if photo_file and allowed_file(photo_file.filename):
            user.photo = save_upload(photo_file, prefix="user-")
        db.session.add(user)
        db.session.commit()
        flash("User berhasil ditambahkan.", "success")
        return redirect(url_for("admin_users"))
    return render_template("user_form.html", user=None)

@app.route("/admin/users/<int:user_id>/edit", methods=["GET", "POST"])
@login_required
@admin_required
def admin_users_edit(user_id):
    user = User.query.get_or_404(user_id)
    if request.method == "POST":
        user.name = request.form.get("name", "").strip()
        user.email = request.form.get("email", "").strip().lower()
        user.gender = request.form.get("gender", "")
        user.phone = request.form.get("phone", "")
        user.position = request.form.get("position", "")
        user.role = request.form.get("role", "user")
        new_password = request.form.get("password", "")
        photo_file = request.files.get("photo")

        if new_password:
            user.set_password(new_password)
        if photo_file and allowed_file(photo_file.filename):
            user.photo = save_upload(photo_file, prefix="user-")

        db.session.commit()
        flash("User berhasil diperbarui.", "success")
        return redirect(url_for("admin_users"))
    return render_template("user_form.html", user=user)

@app.route("/admin/users/<int:user_id>/delete", methods=["POST"])
@login_required
@admin_required
def admin_users_delete(user_id):
    user = User.query.get_or_404(user_id)
    if user.id == current_user.id:
        flash("Tidak dapat menghapus diri sendiri.", "danger")
        return redirect(url_for("admin_users"))
    has_loans = Loan.query.filter_by(user_id=user.id).first() is not None
    if has_loans:
        flash("Tidak dapat menghapus user yang memiliki riwayat pinjaman.", "danger")
        return redirect(url_for("admin_users"))
    db.session.delete(user)
    db.session.commit()
    flash("User berhasil dihapus.", "success")
    return redirect(url_for("admin_users"))

# ---------------- Routes: Admin - Loans ----------------
@app.route("/admin/loans")
@login_required
@admin_required
def admin_loans():
    loans = Loan.query.order_by(Loan.created_at.desc()).all()
    return render_template("loans_index.html", loans=loans)

@app.route("/admin/history")
@login_required
@admin_required
def admin_history():
    loans = Loan.query.order_by(Loan.created_at.desc()).all()
    return render_template("admin_history.html", loans=loans)

@app.route("/admin/loans/<int:loan_id>/approve", methods=["POST"])
@login_required
@admin_required
def admin_loan_approve(loan_id):
    loan = Loan.query.get_or_404(loan_id)
    if loan.status != "requested":
        flash("Hanya pengajuan yang bisa disetujui.", "warning")
        return redirect(url_for("admin_loans"))
    # check availability
    if available_count(loan.item) <= 0:
        flash("Kapasitas barang tidak tersedia.", "danger")
        return redirect(url_for("admin_loans"))
    loan.status = "approved"
    db.session.commit()
    flash("Pengajuan disetujui.", "success")
    return redirect(url_for("admin_loans"))

@app.route("/admin/loans/<int:loan_id>/borrow", methods=["POST"])
@login_required
@admin_required
def admin_loan_borrow(loan_id):
    loan = Loan.query.get_or_404(loan_id)
    if loan.status not in ["approved", "requested"]:
        flash("Hanya pengajuan yang sudah disetujui atau diajukan yang bisa ditandai dipinjam.", "warning")
        return redirect(url_for("admin_loans"))
    borrow_photo = request.files.get("borrow_photo")
    if not borrow_photo or not allowed_file(borrow_photo.filename):
        flash("Foto peminjaman wajib dan harus gambar yang valid.", "danger")
        return redirect(url_for("admin_loans"))
    loan.borrow_photo = save_upload(borrow_photo, prefix="bor-")
    loan.status = "borrowed"
    loan.start_date = date.today()
    db.session.commit()
    flash("Status diubah menjadi Dipinjam.", "success")
    return redirect(url_for("admin_loans"))

@app.route("/admin/loans/<int:loan_id>/complete", methods=["POST"])
@login_required
@admin_required
def admin_loan_complete(loan_id):
    loan = Loan.query.get_or_404(loan_id)
    if loan.status != "borrowed":
        flash("Hanya pinjaman aktif yang bisa diselesaikan.", "warning")
        return redirect(url_for("admin_loans"))
    ret_photo = request.files.get("return_photo")
    if not ret_photo or not allowed_file(ret_photo.filename):
        flash("Foto pengembalian wajib dan harus gambar yang valid.", "danger")
        return redirect(url_for("admin_loans"))
    loan.return_photo = save_upload(ret_photo, prefix="ret-")
    loan.status = "completed"
    loan.end_date = date.today()
    db.session.commit()
    flash("Pinjaman diselesaikan.", "success")
    return redirect(url_for("admin_loans"))

@app.post('/admin/loans/<int:loan_id>/delete')
@login_required
@admin_required
def admin_loan_delete(loan_id):
    loan = Loan.query.get_or_404(loan_id)

    # (Opsional) hapus file foto terkait agar tidak jadi sampah
    for fname in [loan.request_photo, loan.borrow_photo, loan.return_photo]:
        if fname:
            fpath = os.path.join(app.config['UPLOAD_FOLDER'], fname)
            try:
                if os.path.exists(fpath):
                    os.remove(fpath)
            except Exception:
                pass  # jangan ganggu alur kalau gagal hapus file

    db.session.delete(loan)
    db.session.commit()
    flash(f'Pinjaman #{loan.id} dihapus.', 'success')
    return redirect(url_for('admin_loans'))
# ---------------- CLI & bootstrap ----------------
def init_db(make_admin=True):
    db.create_all()
    if make_admin and not User.query.filter_by(email="admin@local").first():
        admin = User(
            name="Administrator",
            email="admin@local",
            gender="Lainnya",
            phone="-",
            position="Admin",
            role="admin",
        )
        admin.set_password(os.environ.get("ADMIN_PASSWORD", "admin123"))
        db.session.add(admin)
        db.session.commit()
        print("Admin default dibuat: admin@local / admin123")

if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--initdb", action="store_true", help="Create DB and seed admin.")
    args = parser.parse_args()
    if args.initdb:
        with app.app_context():
            init_db(make_admin=True)
    else:
        # If DB not exists, create silently
        with app.app_context():
            if not os.path.exists(os.path.join(BASE_DIR, "app.db")):
                init_db(make_admin=True)
        app.run(debug=True)
